<?php

namespace App\Http\Controllers;

use App\Imports\SimpleUsersImport;
use App\Mail\SimpleGreetingMail;
use Illuminate\Support\Str;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Support\Facades\Hash;
use App\Models\UserMaster;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Maatwebsite\Excel\Facades\Excel;
use App\Mail\UserRegistrationMail;

use Throwable;

class UserMasterController extends Controller
{
    public function index(Request $request)
    {
        try {
            $query = UserMaster::with('role', 'parentUser', 'relation', 'package');

            if ($request->has('role_id')) {
                $query->where('role_id', $request->input('role_id'));
            }

            $users = $query->get();

            return response()->json($users, 200);
        } catch (Throwable $e) {
            Log::error('User index error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch users'], 500);
        }
    }


    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'name'     => 'required|string',
                'email'    => 'nullable|email|unique:user_masters,email',
                'mobile'   => ['required', 'max:15', 'unique:user_masters,mobile'],
                'password'    => 'nullable|string',
                'role_id'  => 'nullable|exists:roles,id',
                'image'    => 'nullable|image',
                'user_id'     => 'nullable|exists:user_masters,id',
                'relation_id' => 'nullable|exists:relations,id',
                'U_Status'    => 'nullable|in:0,1,2,3',
                'PackageId' => 'nullable|exists:package_masters,PackageId',
            ]);

            $data['U_Status'] = $data['U_Status'] ?? 1;
            $data['role_id'] = $data['role_id'] ?? 4;

            if ($data['role_id'] == 1 && UserMaster::where('role_id', 1)->exists()) {
                return response()->json([
                    'error' => 'Super Admin already exists',
                ], 422);
            }

            $data['password'] = bcrypt($data['password'] ?? '12345678');
            $user = UserMaster::create($data);

            if ($request->hasFile('image')) {
                $image = $request->file('image');
                $imageName = time() . '_' . $image->getClientOriginalName();
                $image->move(public_path('storage/users/'), $imageName);
                $user->image = url('storage/users/' . $imageName);
                $user->save();
            }
            if (!empty($user->email)) {
                Mail::to($user->email)->send(new UserRegistrationMail([
                    'name' => $user->name,
                    'email' => $user->email
                ]));
            }
            return response()->json(['message' => 'User created', 'user' => $user], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('User store error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create user'], 500);
        }
    }

    public function show($id)
    {
        try {
            $user = UserMaster::with('role', 'parentUser', 'relation', 'package')->find($id);
            if (!$user) {
                return response()->json(['message' => 'User not found'], 404);
            }
            return response()->json($user, 200);
        } catch (Throwable $e) {
            Log::error('User show error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch user'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $user = UserMaster::find($id);
            if (!$user) {
                return response()->json(['message' => 'User not found'], 404);
            }

            $data = $request->validate([
                'name'     => 'sometimes|string',
                'email'    => "nullable|sometimes|email|unique:user_masters,email,{$id}",
                'mobile'   => ['sometimes', 'required', 'max:15', "unique:user_masters,mobile,{$id}"],
                'password' => 'nullable|sometimes|string',
                'role_id'  => 'sometimes|exists:roles,id',
                'image'    => 'nullable|string',
                'user_id'     => 'nullable|sometimes|nullable|exists:user_masters,id',
                'relation_id' => 'nullable|sometimes|nullable|exists:relations,id',
                'U_Status'    => 'nullable|sometimes|in:0,1,2,3',
                'PackageId' => 'nullable|sometimes|exists:package_masters,PackageId',
            ]);

            if ($request->role_id == 1 && UserMaster::where('role_id', 1)->where('id', '!=', $id)->exists()) {
                return response()->json([
                    'error' => 'Validation error',
                    'message' => ['role_id' => ['Super Admin already exists.']]
                ], 422);
            }

            if ($request->filled('image') && Str::startsWith($request->image, 'data:image')) {
                $imageData = $request->image;
                if (!empty($user->image)) {
                    $oldPath = str_replace(url('storage') . '/', '', $user->image);
                    $fullOldPath = public_path('storage/' . $oldPath);
                    if (file_exists($fullOldPath)) {
                        unlink($fullOldPath);
                    }
                }

                preg_match('/^data:image\/(\w+);base64,/', $imageData, $typeMatch);
                $imageType = $typeMatch[1] ?? 'png';
                $imageData = preg_replace('/^data:image\/\w+;base64,/', '', $imageData);
                $imageData = str_replace(' ', '+', $imageData);
                $imageBinary = base64_decode($imageData);

                if ($imageBinary === false) {
                    return response()->json(['error' => 'Invalid base64 image'], 422);
                }

                $imageName = time() . '_' . uniqid() . '.' . $imageType;
                $imagePath = 'users/' . $imageName;
                $fullPath = public_path('storage/' . $imagePath);

                if (!file_exists(public_path('storage/users/'))) {
                    mkdir(public_path('storage/users/'), 0777, true);
                }

                file_put_contents($fullPath, $imageBinary);
                $data['image'] = url('storage/' . $imagePath);
            }

            if (!empty($data['password'])) {
                $data['password'] = bcrypt($data['password']);
            }

            $user->update($data);

            return response()->json([
                'message' => 'User updated successfully',
                'user' => $user,
            ]);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('User update error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update user'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $user = UserMaster::find($id);
            if (!$user) {
                return response()->json(['message' => 'User not found'], 404);
            }

            $user->delete();
            return response()->json(['message' => 'User deleted'], 200);
        } catch (Throwable $e) {
            Log::error('User delete error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete user'], 500);
        }
    }

    public function login(Request $request)
    {
        try {
            $request->validate([
                'identifier' => 'required|string',
                'password'   => 'required|string',
            ]);

            $user = UserMaster::with('role')
                ->where('email', $request->identifier)
                ->orWhere('mobile', $request->identifier)
                ->first();

            if (!$user) {
                return response()->json([
                    'message' => 'Email or mobile number does not exist'
                ], 404);
            }
            if ($user->U_Status == 0) {
                return response()->json([
                    'message' => 'Your account is inactive. Please contact support.'
                ], 403);
            }

            if (!Hash::check($request->password, $user->password)) {
                return response()->json([
                    'message' => 'The password you entered is incorrect'
                ], 401);
            }
            $token = $user->createToken('auth_token')->plainTextToken;

            return response()->json([
                'message' => 'Login successful',
                'token'   => $token,
                'user'    => $user,
            ], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Login error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to login'], 500);
        }
    }

    // public function importUsers(Request $request)
    // {
    //     try {
    //         $request->validate([
    //             'file' => 'required|file|mimes:csv,xlsx,xls',
    //         ]);

    //         if (!class_exists('ZipArchive')) {
    //             Log::error('ZipArchive class not found. Ensure the PHP zip extension is enabled.');
    //             return response()->json([
    //                 'error' => 'Server configuration error: PHP zip extension is not enabled.',
    //             ], 500);
    //         }

    //         Excel::import(new SimpleUsersImport, $request->file('file'));

    //         return response()->json([
    //             'message' => 'Users imported successfully',
    //         ], 200);
    //     } catch (ValidationException $e) {
    //         Log::error('Validation error during user import: ' . json_encode($e->errors()));
    //         return response()->json(['errors' => $e->errors()], 422);
    //     } catch (\Throwable $e) {
    //         Log::error('Excel user import error: ' . $e->getMessage());
    //         return response()->json([
    //             'error' => 'Failed to import users',
    //             'details' => $e->getMessage()
    //         ], 500);
    //     }
    // }


    public function importUsers(Request $request)
    {
        try {
            $request->validate([
                'file' => 'required|file|mimes:csv,xlsx,xls',
            ]);

            // Excel::import(new SimpleUsersImport, $request->file('file'));

            // return response()->json([
            //     'message' => 'Users imported successfully',
            // ], 200);
            $import = new SimpleUsersImport;
            Excel::import($import, $request->file('file'));

            return response()->json([
                'message' => 'Users imported successfully.',
                // 'skipped_rows' => $import->failures()->count(),  
            ], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (\Throwable $e) {
            Log::error('Excel user import error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to import users', 'details' => $e->getMessage()], 500);
        }
    }
}
