<?php

namespace App\Http\Controllers;

use App\Models\TpsMemrep;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class TpsMemrepController extends Controller
{
    public function index()
    {
        try {
            $data = TpsMemrep::with(['taskMember', 'taskPlanner', 'suggestion', 'answer'])->get();
            return response()->json($data, 200);
        } catch (Throwable $e) {
            Log::error('Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch records'], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'task_member_id'    => 'nullable|exists:task_members,id',
                'task_planner_id'   => 'nullable|exists:task_planner,id',
                'tp_suggestion_id'  => 'nullable|exists:tp_suggestions,id',
                'tpq_answer_id'     => 'nullable|exists:tpq_answers,id',
            ]);

            $record = TpsMemrep::create($data);
            return response()->json(['message' => 'Created', 'data' => $record], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create record'], 500);
        }
    }

    public function show($id)
    {
        try {
            $record = TpsMemrep::with(['taskMember', 'taskPlanner', 'suggestion', 'answer'])->find($id);
            if (!$record) return response()->json(['message' => 'Not found'], 404);

            return response()->json($record, 200);
        } catch (Throwable $e) {
            Log::error('Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch record'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $record = TpsMemrep::find($id);
            if (!$record) return response()->json(['message' => 'Not found'], 404);

            $data = $request->validate([
                'task_member_id'    => 'sometimes|nullable|exists:task_members,id',
                'task_planner_id'   => 'sometimes|nullable|exists:task_planner,id',
                'tp_suggestion_id'  => 'sometimes|nullable|exists:tp_suggestions,id',
                'tpq_answer_id'     => 'sometimes|nullable|exists:tpq_answers,id',
            ]);

            $record->update($data);
            return response()->json(['message' => 'Updated', 'data' => $record], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update record'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $record = TpsMemrep::find($id);
            if (!$record) return response()->json(['message' => 'Not found'], 404);

            $record->delete();
            return response()->json(['message' => 'Deleted'], 200);
        } catch (Throwable $e) {
            Log::error('Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete record'], 500);
        }
    }
}
