<?php

namespace App\Http\Controllers;

use App\Models\TpqAnswer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class TpqAnswerController extends Controller
{
    public function index()
    {
        try {
            return response()->json(TpqAnswer::with('suggestion')->get(), 200);
        } catch (Throwable $e) {
            Log::error('Answer Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch answers'], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'tp_suggestion_id' => 'nullable|exists:tp_suggestions,id',
                'TP_Answer'        => 'required|string',
                'TPA_Image'        => 'nullable|array', // Accept array of files
                'TPA_Image.*'      => 'image',           // Each must be a valid image
                'TPA_Option'       => 'nullable|integer',
            ]);

            $imageUrls = [];

            if ($request->hasFile('TPA_Image')) {
                foreach ($request->file('TPA_Image') as $image) {
                    $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                    $image->move(public_path('storage/answer_images'), $imageName);
                    $imageUrls[] = url('storage/answer_images/' . $imageName);
                }
            }

            $data['TPA_Image'] = $imageUrls;

            $answer = TpqAnswer::create($data);
            return response()->json(['message' => 'Answer created', 'data' => $answer], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Answer Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create answer'], 500);
        }
    }

    public function show($id)
    {
        try {
            $answer = TpqAnswer::with('suggestion')->find($id);
            if (!$answer) return response()->json(['message' => 'Answer not found'], 404);

            return response()->json($answer, 200);
        } catch (Throwable $e) {
            Log::error('Answer Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch answer'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $answer = TpqAnswer::find($id);
            if (!$answer) return response()->json(['message' => 'Answer not found'], 404);

            $data = $request->validate([
                'tp_suggestion_id' => 'sometimes|nullable|exists:tp_suggestions,id',
                'TP_Answer'        => 'sometimes|nullable|string',
                'TPA_Image'        => 'sometimes|nullable|array',  // Accept array of base64 images
                'TPA_Image.*'      => 'sometimes|nullable|string', // Base64 images
                'TPA_Option'       => 'sometimes|nullable|integer',
            ]);

            $imageUrls = is_array($answer->TPA_Image)
                ? $answer->TPA_Image
                : (json_decode($answer->TPA_Image, true) ?: []);

            // If new base64 images are provided
            if ($request->filled('TPA_Image')) {
                $imageUrls = []; // Reset image URLs
                foreach ($request->input('TPA_Image') as $img) {
                    if (preg_match('/^data:image\/(\w+);base64,/', $img)) {
                        $extension = preg_match('/^data:image\/(\w+);base64,/', $img, $matches) ? $matches[1] : 'png';
                        $img = preg_replace('/^data:image\/\w+;base64,/', '', $img);
                        $img = str_replace(' ', '+', $img);
                        $imageBinary = base64_decode($img);

                        if ($imageBinary === false) {
                            return response()->json(['error' => 'Invalid base64 image'], 422);
                        }

                        $filename = time() . '_' . uniqid() . '.' . $extension;
                        $path = public_path('storage/answer_images/' . $filename);
                        file_put_contents($path, $imageBinary);

                        $imageUrls[] = url('storage/answer_images/' . $filename);
                    }
                }
            }

            $data['TPA_Image'] = $imageUrls;

            $answer->update($data);
            return response()->json(['message' => 'Answer updated', 'data' => $answer], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Answer Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update answer'], 500);
        }
    }


    public function destroy($id)
    {
        try {
            $answer = TpqAnswer::find($id);
            if (!$answer) return response()->json(['message' => 'Answer not found'], 404);

            $answer->delete();
            return response()->json(['message' => 'Answer deleted'], 200);
        } catch (Throwable $e) {
            Log::error('Answer Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete answer'], 500);
        }
    }

    // // ───── Helper Function for Base64 Image ─────
    // private function processBase64Image($base64Image)
    // {
    //     preg_match('/^data:image\/(\w+);base64,/', $base64Image, $typeMatch);
    //     $imageType = $typeMatch[1] ?? 'png';
    //     $imageData = base64_decode(preg_replace('/^data:image\/\w+;base64,/', '', $base64Image));

    //     $filename = time() . '_' . uniqid() . '.' . $imageType;
    //     $path = public_path('storage/answer_images/');
    //     if (!file_exists($path)) mkdir($path, 0777, true);

    //     file_put_contents($path . $filename, $imageData);
    //     return url('storage/answer_images/' . $filename);
    // }
}
