<?php

namespace App\Http\Controllers;

use App\Models\TpSuggestion;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class TpSuggestionController extends Controller
{
    public function index()
    {
        try {
            $suggestions = TpSuggestion::with('taskPlanner')->get();
            return response()->json($suggestions);
        } catch (Throwable $e) {
            Log::error('TP Suggestion Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch suggestions'], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'TPS_Question'     => 'required|string',
                'TPS_Image'        => 'nullable|array',
                'TPS_Image.*'      => 'image', // multiple image upload support
                'task_planner_id'  => 'nullable|exists:task_planner,id',
                'TPS_Status'       => 'nullable|in:0,1,2',
            ]);

            $imageUrls = [];

            // Store multiple images
            if ($request->hasFile('TPS_Image')) {
                foreach ($request->file('TPS_Image') as $image) {
                    $imageName = time() . '_' . uniqid() . '.' . $image->getClientOriginalExtension();
                    $image->move(public_path('storage/tp_suggestions'), $imageName);
                    $imageUrls[] = url('storage/tp_suggestions/' . $imageName);
                }
            }

            $data['TPS_Image'] = $imageUrls;

            $suggestion = TpSuggestion::create($data);

            return response()->json(['message' => 'Suggestion created', 'data' => $suggestion], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('TP Suggestion Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create suggestion'], 500);
        }
    }


    public function show($id)
    {
        try {
            $suggestion = TpSuggestion::with('taskPlanner')->find($id);
            if (!$suggestion) return response()->json(['message' => 'Suggestion not found'], 404);

            return response()->json($suggestion);
        } catch (Throwable $e) {
            Log::error('TP Suggestion Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch suggestion'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $suggestion = TpSuggestion::find($id);
            if (!$suggestion) {
                return response()->json(['message' => 'Suggestion not found'], 404);
            }

            $data = $request->validate([
                'TPS_Question'     => 'nullable|string',
                'TPS_Image'        => 'nullable|array',
                'TPS_Image.*'      => 'nullable|string', // base64 images
                'task_planner_id'  => 'nullable|exists:task_planner,id',
                'TPS_Status'       => 'nullable|in:0,1,2',
            ]);

            $imageUrls = is_array($suggestion->TPS_Image)
                ? $suggestion->TPS_Image
                : (json_decode($suggestion->TPS_Image, true) ?: []);

            // If new base64 images are provided
            if ($request->filled('TPS_Image')) {
                $imageUrls = []; // reset to accept new ones
                foreach ($request->input('TPS_Image') as $img) {
                    if (preg_match('/^data:image\/(\w+);base64,/', $img)) {
                        $extension = preg_match('/^data:image\/(\w+);base64,/', $img, $matches) ? $matches[1] : 'png';
                        $img = preg_replace('/^data:image\/\w+;base64,/', '', $img);
                        $img = str_replace(' ', '+', $img);
                        $imageBinary = base64_decode($img);

                        if ($imageBinary === false) {
                            return response()->json(['error' => 'Invalid base64 image'], 422);
                        }

                        $filename = time() . '_' . uniqid() . '.' . $extension;
                        $path = public_path('storage/tp_suggestions/' . $filename);
                        file_put_contents($path, $imageBinary);

                        $imageUrls[] = url('storage/tp_suggestions/' . $filename);
                    }
                }
            }

            $data['TPS_Image'] = $imageUrls;

            $suggestion->update($data);

            return response()->json(['message' => 'Suggestion updated', 'data' => $suggestion], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('TP Suggestion Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update suggestion'], 500);
        }
    }


    public function destroy($id)
    {
        try {
            $suggestion = TpSuggestion::find($id);
            if (!$suggestion) return response()->json(['message' => 'Suggestion not found'], 404);

            $suggestion->delete();
            return response()->json(['message' => 'Suggestion deleted'], 200);
        } catch (Throwable $e) {
            Log::error('TP Suggestion Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete suggestion'], 500);
        }
    }

    // Optional: Base64 image processing
    private function processBase64Image($base64Image)
    {
        preg_match('/^data:image\/(\w+);base64,/', $base64Image, $matches);
        $type = $matches[1] ?? 'png';
        $image = base64_decode(preg_replace('/^data:image\/\w+;base64,/', '', $base64Image));
        $filename = time() . '_' . uniqid() . '.' . $type;
        $path = public_path('storage/tp_suggestions/');
        if (!file_exists($path)) mkdir($path, 0777, true);
        file_put_contents($path . $filename, $image);

        return url('storage/tp_suggestions/' . $filename);
    }
}
