<?php

namespace App\Http\Controllers;

use App\Models\TaskPlanner;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class TaskPlannerController extends Controller
{
    public function index()
    {
        try {
            $planners = TaskPlanner::with(['evTask', 'planner', 'taskMember', 'evTask.event', 'taskMember.eventMember'])->get();

            $planners->each(function ($planner) {
                $taskMember = $planner->taskMember;

                // Ensure tp_user_statuses is included and properly formatted
                $tpUserStatuses = collect(
                    is_array($planner->tp_user_statuses)
                        ? $planner->tp_user_statuses
                        : (json_decode($planner->tp_user_statuses, true) ?: [])
                );

                // Log for debugging
                Log::debug('Planner ID: ' . $planner->id . ' - tp_user_statuses: ', $tpUserStatuses->toArray());

                // Set tp_user_statuses attribute
                $planner->setAttribute('tp_user_statuses', $tpUserStatuses->values()->toArray());

                if ($taskMember) {
                    $assignedUserIds = $taskMember->user_id ?? [];

                    // Filter user_statuses from taskMember (optional use internally)
                    $filteredTMStatuses = collect($taskMember->user_statuses ?? [])
                        ->filter(fn($status) => in_array($status['user_id'], $assignedUserIds))
                        ->values()
                        ->all();

                    $taskMember->setAttribute('user_statuses', $filteredTMStatuses);
                    $taskMember->setAttribute('user_id', $assignedUserIds);

                    // Load users for this planner
                    $users = [];
                    foreach ($assignedUserIds as $userId) {
                        $user = \App\Models\UserMaster::find($userId);
                        if ($user) {
                            // Optionally add TP_Status from tp_user_statuses
                            $tpStatus = $tpUserStatuses->firstWhere('user_id', $userId);
                            $user->TP_Status = $tpStatus['TP_Status'] ?? 0;
                            $users[] = $user;
                        }
                    }

                    $planner->setAttribute('users', $users);

                    // Add event member user_detail
                    $eventMember = $taskMember->eventMember;
                    if ($eventMember) {
                        $originalUserData = $eventMember->getRawOriginal('user_id');
                        $decodedUsers = is_string($originalUserData)
                            ? json_decode($originalUserData, true)
                            : $originalUserData;

                        $userDetail = collect($decodedUsers ?? [])->map(function ($entry) {
                            return is_array($entry)
                                ? [
                                    'user_id'   => $entry['user_id'] ?? null,
                                    'UM_Status' => $entry['UM_Status'] ?? 0,
                                ]
                                : ['user_id' => $entry, 'UM_Status' => 0];
                        });

                        $flatUserIds = $userDetail->pluck('user_id')->filter()->values();

                        $eventMember->setAttribute('user_detail', $userDetail);
                        $eventMember->setAttribute('user_id', $flatUserIds);
                    }
                } else {
                    $planner->setAttribute('users', []);
                }
            });

            return response()->json($planners, 200);
        } catch (Throwable $e) {
            Log::error('TaskPlanner Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch task planners'], 500);
        }
    }
    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'ev_task_id'      => 'nullable|exists:ev_tasks,id',
                'TP_Name'         => 'nullable|string',
                'TP_Type'         => 'nullable|string',
                'TP_TimeLine'     => 'nullable|date',
                'TP_Status'       => 'nullable|in:0,1,2',
                'task_member_id'  => 'nullable|exists:task_members,id',
                'tp_user_statuses' => 'sometimes|array',
                'tp_user_statuses.*.user_id' => 'nullable|exists:user_masters,id',
                'tp_user_statuses.*.TP_Status' => 'nullable|in:0,1,2,3',
                'plan_id'          => 'nullable|exists:planners,id', // ✅ Added

            ]);

            $task = TaskPlanner::create($data);

            // Fetch task member users
            $taskMember = \App\Models\TaskMember::find($task->task_member_id);
            $assignedUserIds = $taskMember?->user_id ?? [];

            // Load incoming statuses
            $incomingStatuses = collect($request->tp_user_statuses ?? [])->keyBy('user_id');

            // Build statuses for all assigned users
            $allStatuses = collect($assignedUserIds)->map(function ($userId) use ($incomingStatuses) {
                return [
                    'user_id' => $userId,
                    'TP_Status' => $incomingStatuses[$userId]['TP_Status'] ?? 0,
                ];
            })->values();

            $task->tp_user_statuses = $allStatuses;
            $task->save();

            return response()->json(['message' => 'Created', 'data' => $task], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Store error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create record'], 500);
        }
    }


    public function show($id)
    {
        try {
            $planner = TaskPlanner::with(['evTask', 'taskMember', 'planner', 'evTask.event', 'taskMember.eventMember'])->find($id);

            if (!$planner) {
                return response()->json(['message' => 'Task planner not found'], 404);
            }

            return response()->json($planner, 200);
        } catch (Throwable $e) {
            Log::error('TaskPlanner Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch task planner'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $task = TaskPlanner::find($id);
            if (!$task) return response()->json(['message' => 'Not found'], 404);

            $data = $request->validate([
                'ev_task_id'       => 'sometimes|nullable|exists:ev_tasks,id',
                'TP_Name'          => 'sometimes|nullable|string',
                'TP_Type'          => 'sometimes|nullable|string',
                'TP_TimeLine'      => 'sometimes|nullable|date',
                'TP_Status'        => 'sometimes|nullable|in:0,1,2',
                'task_member_id'   => 'sometimes|nullable|exists:task_members,id',
                'tp_user_statuses' => 'sometimes|array',
                'tp_user_statuses.*.user_id'    => 'sometimes|nullable|exists:user_masters,id',
                'tp_user_statuses.*.TP_Status'  => 'sometimes|nullable|in:0,1,2,3',
                'plan_id'          => 'sometimes|nullable|exists:planners,id', // ✅ Added

            ]);

            // Update fillable fields (excluding tp_user_statuses for now)
            $task->update(array_diff_key($data, array_flip(['tp_user_statuses'])));

            // Handle user statuses
            if ($request->has('tp_user_statuses')) {
                // Load existing statuses from the database
                $existingStatuses = collect(
                    is_array($task->tp_user_statuses)
                        ? $task->tp_user_statuses
                        : (json_decode($task->tp_user_statuses, true) ?: [])
                )->keyBy('user_id');

                // Load incoming statuses from the request
                $incomingStatuses = collect($request->tp_user_statuses ?? [])->keyBy('user_id');

                // Log for debugging
                Log::debug('Existing Statuses Before Update: ', $existingStatuses->toArray());
                Log::debug('Incoming Statuses: ', $incomingStatuses->toArray());

                // Update only the statuses provided in the incoming request, preserve others
                foreach ($incomingStatuses as $userId => $entry) {
                    $existingStatuses[$userId] = [
                        'user_id'   => $userId,
                        'TP_Status' => $entry['TP_Status'] ?? ($existingStatuses[$userId]['TP_Status'] ?? 0),
                    ];
                }

                // Ensure all existing statuses are preserved
                $task->tp_user_statuses = $existingStatuses->values()->toArray();
                $task->save();

                // Log the final statuses
                Log::debug('Updated Statuses: ', $task->tp_user_statuses);
            }

            return response()->json(['message' => 'Updated', 'data' => $task], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Update error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update record'], 500);
        }
    }
    public function destroy($id)
    {
        try {
            $task = TaskPlanner::find($id);
            if (!$task) return response()->json(['message' => 'Not found'], 404);

            $task->delete();
            return response()->json(['message' => 'Deleted'], 200);
        } catch (Throwable $e) {
            Log::error('Delete error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete record'], 500);
        }
    }
}
