<?php

namespace App\Http\Controllers;

use App\Models\TaskMember;
use App\Models\UserMaster;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class TaskMemberController extends Controller
{
    // public function index()
    // {
    //     try {
    //         $members = TaskMember::with(['eventMember', 'task', 'task.event'])->get();

    //         $members->each(function ($member) {
    //             $eventMember = $member->eventMember;
    //             $assignedUserIds = $member->user_id ?? [];

    //             // ✅ Filter only selected user_statuses
    //             $filteredStatuses = collect($member->user_statuses ?? [])
    //                 ->filter(function ($status) use ($assignedUserIds) {
    //                     return in_array($status['user_id'], $assignedUserIds);
    //                 })
    //                 ->values();

    //             $member->setAttribute('user_statuses', $filteredStatuses);

    //             if ($eventMember) {
    //                 $originalUserData = $eventMember->getRawOriginal('user_id');
    //                 $decodedUsers = is_string($originalUserData)
    //                     ? json_decode($originalUserData, true)
    //                     : $originalUserData;

    //                 // ✅ Support both int and array user_id formats
    //                 $userDetail = collect($decodedUsers ?? [])->map(function ($user) {
    //                     return [
    //                         'user_id'   => is_array($user) ? $user['user_id'] : $user,
    //                         'UM_Status' => is_array($user) ? ($user['UM_Status'] ?? 0) : 0,
    //                     ];
    //                 });

    //                 $eventMember->setAttribute('user_detail', $userDetail);
    //                 $eventMember->setAttribute('user_id', $userDetail->pluck('user_id')->values());

    //                 // ✅ Only show assigned users
    //                 $users = $userDetail
    //                     ->filter(fn($u) => in_array($u['user_id'], $assignedUserIds))
    //                     ->map(function ($userData) use ($filteredStatuses) {
    //                         $user = \App\Models\UserMaster::find($userData['user_id']);
    //                         if ($user) {
    //                             $user->UM_Status = $userData['UM_Status'];
    //                             $user->assigned = true;
    //                             $status = $filteredStatuses->firstWhere('user_id', $user->id);
    //                             $user->TM_Status = $status['TM_Status'] ?? 0;
    //                             return $user;
    //                         }
    //                         return null;
    //                     })
    //                     ->filter()
    //                     ->values();

    //                 $member->setAttribute('users', $users);
    //             } else {
    //                 $member->setAttribute('users', collect());
    //                 $member->setRelation('eventMember', null);
    //             }
    //         });

    //         return response()->json($members);
    //     } catch (\Throwable $e) {
    //         Log::error('TaskMember Index Error: ' . $e->getMessage());
    //         return response()->json(['error' => 'Failed to fetch task members'], 500);
    //     }
    // }
    //above changes

    public function index()
    {
        try {
            $members = TaskMember::with(['eventMember', 'task', 'task.event'])->get();

            $members->each(function ($member) {
                $eventMember = $member->eventMember;
                $assignedUserIds = $member->user_id ?? [];

                // ✅ Filter only selected user_statuses (based on assigned user_ids)
                $filteredStatuses = collect($member->user_statuses ?? [])
                    ->filter(function ($status) use ($assignedUserIds) {
                        return in_array($status['user_id'], $assignedUserIds);
                    })
                    ->values();

                $member->setAttribute('user_statuses', $filteredStatuses);

                if ($eventMember) {
                    $originalUserData = $eventMember->getRawOriginal('user_id');
                    $decodedUsers = is_string($originalUserData)
                        ? json_decode($originalUserData, true)
                        : $originalUserData;

                    // ✅ Decode and filter only UM_Status = 1 users
                    $userDetail = collect($decodedUsers ?? [])->map(function ($user) {
                        return [
                            'user_id'   => is_array($user) ? $user['user_id'] : $user,
                            'UM_Status' => is_array($user) ? ($user['UM_Status'] ?? 0) : 0,
                        ];
                    })->filter(fn($u) => $u['UM_Status'] == 1)->values();

                    // ✅ Set filtered user detail and user_id list
                    $eventMember->setAttribute('user_detail', $userDetail);
                    $eventMember->setAttribute('user_id', $userDetail->pluck('user_id')->values());

                    // ✅ Load only users that are assigned and UM_Status = 1
                    $users = $userDetail
                        ->filter(fn($u) => in_array($u['user_id'], $assignedUserIds))
                        ->map(function ($userData) use ($filteredStatuses) {
                            $user = \App\Models\UserMaster::find($userData['user_id']);
                            if ($user) {
                                $user->UM_Status = $userData['UM_Status'];
                                $user->assigned = true;
                                $status = $filteredStatuses->firstWhere('user_id', $user->id);
                                $user->TM_Status = $status['TM_Status'] ?? 0;
                                return $user;
                            }
                            return null;
                        })
                        ->filter()
                        ->values();

                    $member->setAttribute('users', $users);
                } else {
                    $member->setAttribute('users', collect());
                    $member->setRelation('eventMember', null);
                }
            });

            return response()->json($members);
        } catch (\Throwable $e) {
            Log::error('TaskMember Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch task members'], 500);
        }
    }








    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'event_member_id' => 'required|exists:event_members,id',
                'ev_task_id'      => 'required|exists:ev_tasks,id',
                'Task_Invite'     => 'nullable|date',
                'Task_Accept'     => 'nullable|date',
                'TM_Status'       => 'nullable|in:0,1,2',
                'user_id'         => 'nullable|array',
                'user_id.*'       => 'nullable|exists:user_masters,id',
                'user_status'     => 'nullable|array',
                'user_status.*.user_id' => 'nullable|exists:user_masters,id',
                'user_status.*.TM_Status' => 'nullable|in:0,1,2',
            ]);

            $data['Task_Invite'] = $data['Task_Invite'] ?? Carbon::now();

            if (isset($data['TM_Status']) && $data['TM_Status'] == 1 && empty($data['Task_Accept'])) {
                $data['Task_Accept'] = Carbon::now();
            }

            $member = TaskMember::create($data);
            $member->user_id = $data['user_id'] ?? [];

            // Handle user_status
            if ($request->has('user_status')) {
                $userStatuses = collect($request->user_status)->map(function ($entry) {
                    return [
                        'user_id' => $entry['user_id'],
                        'TM_Status' => $entry['TM_Status'] ?? 0,
                    ];
                });
            } else {
                $eventMember = $member->eventMember;
                $userStatuses = collect($eventMember->user_id ?? [])->map(function ($user) {
                    return [
                        'user_id' => $user['user_id'],
                        'TM_Status' => 0,
                    ];
                });
            }

            $member->user_statuses = $userStatuses;
            $member->save();

            return response()->json(['message' => 'Task member created', 'data' => $member], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create task member'], 500);
        }
    }

    public function show($id)
    {
        try {
            $member = TaskMember::with(['eventMember', 'task', 'task.event'])->find($id);
            if (!$member) return response()->json(['message' => 'Not found'], 404);

            $eventMember = $member->eventMember;
            $userStatuses = collect($member->user_statuses ?? []);
            $assignedUserIds = $member->user_id ?? [];

            if ($eventMember) {
                // Decode original user_id JSON
                $originalUserData = $eventMember->getRawOriginal('user_id');
                $decodedUsers = is_string($originalUserData) ? json_decode($originalUserData, true) : $originalUserData;

                $userDetail = collect($decodedUsers ?? [])->map(function ($user) {
                    return [
                        'user_id'   => $user['user_id'],
                        'UM_Status' => $user['UM_Status'] ?? 0,
                    ];
                });

                $eventMember->setAttribute('user_detail', $userDetail);
                $eventMember->setAttribute('user_id', $userDetail->pluck('user_id')->values());

                // Filter only selected assigned users
                $users = $userDetail->filter(function ($userData) use ($assignedUserIds) {
                    return in_array($userData['user_id'], $assignedUserIds);
                })->map(function ($userData) use ($userStatuses) {
                    $user = UserMaster::find($userData['user_id']);
                    if ($user) {
                        $user->UM_Status = $userData['UM_Status'];
                        $user->assigned = true;
                        $status = $userStatuses->firstWhere('user_id', $user->id);
                        $user->TM_Status = $status['TM_Status'] ?? 0;
                        return $user;
                    }
                    return null;
                })->filter()->values();

                $member->setAttribute('users', $users);
            } else {
                $member->setAttribute('users', collect());
                $member->setRelation('eventMember', null);
            }

            return response()->json($member);
        } catch (Throwable $e) {
            Log::error('Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch data'], 500);
        }
    }


    public function update(Request $request, $id)
    {
        try {
            $member = TaskMember::find($id);
            if (!$member) return response()->json(['message' => 'Not found'], 404);

            $data = $request->validate([
                'event_member_id' => 'sometimes|exists:event_members,id',
                'ev_task_id'      => 'sometimes|exists:ev_tasks,id',
                'Task_Invite'     => 'sometimes|nullable|date',
                'Task_Accept'     => 'sometimes|nullable|date',
                'TM_Status'       => 'sometimes|nullable|in:0,1,2',
                'user_id'         => 'sometimes|nullable|array',
                'user_id.*'       => 'sometimes|nullable|exists:user_masters,id',
                'user_status'     => 'sometimes|nullable|array',
                'user_status.*.user_id' => 'sometimes|nullable|exists:user_masters,id',
                'user_status.*.TM_Status' => 'sometimes|nullable|required|in:0,1,2',
            ]);

            if (empty($data['Task_Invite'])) {
                $data['Task_Invite'] = Carbon::now();
            }

            if (
                isset($data['TM_Status']) && $data['TM_Status'] == 1 &&
                empty($data['Task_Accept'])
            ) {
                $data['Task_Accept'] = Carbon::now();
            }

            $member->update($data);

            if ($request->has('user_id')) {
                $member->user_id = $data['user_id'];
                $member->save();

                // Auto-set statuses if not manually passed
                if (!$request->has('user_status')) {
                    $member->user_statuses = collect($data['user_id'])->map(function ($id) {
                        return [
                            'user_id'   => $id,
                            'TM_Status' => 0
                        ];
                    });
                    $member->save();
                }
            }

            // ✅ Update only incoming user statuses — do not overwrite others
            if ($request->has('user_status')) {
                $existingStatuses = collect($member->user_statuses ?? [])->keyBy('user_id');
                $incomingStatuses = collect($request->user_status)->keyBy('user_id');

                foreach ($incomingStatuses as $userId => $entry) {
                    $existingStatuses[$userId] = [
                        'user_id'   => $userId,
                        'TM_Status' => $entry['TM_Status'] ?? 0
                    ];
                }

                $member->user_statuses = $existingStatuses->values();
                $member->save();
            }

            return response()->json(['message' => 'Updated', 'data' => $member]);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update task member'], 500);
        }
    }


    public function destroy($id)
    {
        try {
            $member = TaskMember::find($id);
            if (!$member) return response()->json(['message' => 'Not found'], 404);

            $member->delete();
            return response()->json(['message' => 'Deleted']);
        } catch (Throwable $e) {
            Log::error('Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete task member'], 500);
        }
    }
}
