<?php

namespace App\Http\Controllers;

use App\Models\TaskGallary;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class TaskGallaryController extends Controller
{

    /* ───── GET ALL ───── */
    public function index()
    {
        try {
            return response()->json(TaskGallary::with('taskMember')->get(), 200);
        } catch (Throwable $e) {
            Log::error('TaskGallary Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch records'], 500);
        }
    }

    /* ───── CREATE ───── */
    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'task_member_id' => 'nullable|exists:task_members,id',
                'Image'          => 'nullable|array',
                'Image.*'        => 'image',
                'Img_Date'       => 'nullable|date',
                'Img_GPS'        => 'nullable|string',
            ]);

            $imageUrls = [];

            if ($request->hasFile('Image')) {
                foreach ($request->file('Image') as $img) {
                    $name = time() . '_' . uniqid() . '.' . $img->getClientOriginalExtension();
                    $img->move(public_path('storage/task_gallery'), $name);
                    $imageUrls[] = url('storage/task_gallery/' . $name);
                }
            }

            $data['Image'] = $imageUrls;   // store as JSON array

            $gallery = TaskGallary::create($data);
            return response()->json(['message' => 'Image(s) saved', 'data' => $gallery], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('TaskGallary Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to save image(s)'], 500);
        }
    }

    /* ───── SHOW ONE ───── */
    public function show($id)
    {
        try {
            $gallery = TaskGallary::with('taskMember')->find($id);
            if (!$gallery) return response()->json(['message' => 'Record not found'], 404);
            return response()->json($gallery, 200);
        } catch (Throwable $e) {
            Log::error('TaskGallary Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Fetch failed'], 500);
        }
    }

    /* ───── UPDATE ───── */
    public function update(Request $request, $id)
    {
        try {
            $gallery = TaskGallary::find($id);
            if (!$gallery) return response()->json(['message' => 'Not found'], 404);

            $data = $request->validate([
                'task_member_id' => 'sometimes|nullable|exists:task_members,id',
                'Image'   => 'sometimes',                  // single base64 or array

                'Img_Date'       => 'sometimes|nullable|date',
                'Img_GPS'        => 'sometimes|nullable|string',
            ]);

            // Load current image array
            $imageUrls = is_array($gallery->Image)
                ? $gallery->Image
                : (json_decode($gallery->Image, true) ?: []);


            /* ───── UPDATE IMAGE(S) ───── */
            if (isset($data['Image'])) {
                if (is_string($data['Image']) && count($imageUrls) === 1) {
                    // Update only image (no index needed)
                    $b64 = $data['Image'];
                    if (preg_match('/^data:image\/(\w+);base64,/', $b64, $m)) {
                        $ext = $m[1];
                        $bin = base64_decode(preg_replace('/^data:image\/\w+;base64,/', '', $b64));
                        if ($bin === false) return response()->json(['error' => 'Invalid base64 image'], 422);

                        $fname = time() . '_' . uniqid() . '.' . $ext;
                        $fpath = public_path('storage/task_gallery/' . $fname);
                        if (!file_exists(dirname($fpath))) mkdir(dirname($fpath), 0777, true);
                        file_put_contents($fpath, $bin);
                        $imageUrls[0] = url('storage/task_gallery/' . $fname);
                    }
                } elseif (is_array($data['Image'])) {
                    foreach ($data['Image'] as $index => $b64) {
                        if (preg_match('/^data:image\/(\w+);base64,/', $b64, $m)) {
                            $ext = $m[1];
                            $bin = base64_decode(preg_replace('/^data:image\/\w+;base64,/', '', $b64));
                            if ($bin === false) return response()->json(['error' => "Invalid base64 image at index $index"], 422);

                            $fname = time() . '_' . uniqid() . '.' . $ext;
                            $fpath = public_path('storage/task_gallery/' . $fname);
                            if (!file_exists(dirname($fpath))) mkdir(dirname($fpath), 0777, true);
                            file_put_contents($fpath, $bin);
                            $imageUrls[$index] = url('storage/task_gallery/' . $fname);
                        }
                    }
                }
            }



            $data['Image'] = array_values($imageUrls); // ensure reindex

            $gallery->update($data);

            return response()->json(['message' => 'Gallery updated', 'data' => $gallery], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('TaskGallary Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Update failed'], 500);
        }
    }



    /* ───── DELETE ───── */
    public function destroy($id)
    {
        try {
            $gallery = TaskGallary::find($id);
            if (!$gallery) return response()->json(['message' => 'Not found'], 404);

            $gallery->delete();
            return response()->json(['message' => 'Deleted'], 200);
        } catch (Throwable $e) {
            Log::error('TaskGallary Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete'], 500);
        }
    }


    public function deleteImage($id, $index)
    {
        try {
            $gallery = TaskGallary::find($id);
            if (!$gallery) {
                return response()->json(['error' => 'Gallery not found'], 404);
            }

            $images = is_array($gallery->Image)
                ? $gallery->Image
                : (json_decode($gallery->Image, true) ?: []);

            if (!isset($images[$index])) {
                return response()->json(['error' => 'Image index not found'], 404);
            }

            // Delete the actual file from disk
            $localPath = str_replace(url('/'), public_path(), $images[$index]);
            if (file_exists($localPath)) {
                @unlink($localPath);
            }

            // Remove image from array
            unset($images[$index]);
            $gallery->Image = array_values($images); // Reindex array
            $gallery->save();

            return response()->json(['message' => 'Image deleted successfully', 'images' => $gallery->Image], 200);
        } catch (\Throwable $e) {
            Log::error('Delete Image Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete image'], 500);
        }
    }


    public function updateImageByIndex(Request $request, $id, $index)
    {
        try {
            $request->validate([
                'Image' => 'required|string' // base64 string
            ]);

            $gallery = TaskGallary::find($id);
            if (!$gallery) {
                return response()->json(['error' => 'Gallery not found'], 404);
            }

            $Images = is_array($gallery->Image)
                ? $gallery->Image
                : (json_decode($gallery->Image, true) ?: []);

            if (!isset($Images[$index])) {
                return response()->json(['error' => 'Image index not found'], 404);
            }

            $b64 = $request->Image;

            if (!preg_match('/^data:image\/(\w+);base64,/', $b64, $m)) {
                return response()->json(['error' => 'Invalid base64 image format'], 422);
            }

            $ext = $m[1];
            $bin = base64_decode(preg_replace('/^data:image\/\w+;base64,/', '', $b64));
            if ($bin === false) {
                return response()->json(['error' => 'Invalid base64 image data'], 422);
            }

            // Remove old image from disk
            $oldPath = str_replace(url('/'), public_path(), $Images[$index]);
            if (file_exists($oldPath)) {
                @unlink($oldPath);
            }

            // Save new image
            $fname = time() . '_' . uniqid() . '.' . $ext;
            $fpath = public_path('storage/task_gallery/' . $fname);
            if (!file_exists(dirname($fpath))) mkdir(dirname($fpath), 0777, true);
            file_put_contents($fpath, $bin);

            // Update array
            $Images[$index] = url('storage/task_gallery/' . $fname);
            $gallery->Image = array_values($Images);
            $gallery->save();

            return response()->json([
                'message' => 'Image updated successfully',
                'images' => $gallery->Image
            ], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (\Throwable $e) {
            Log::error('Update Image Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update image'], 500);
        }
    }



    public function addImages(Request $request, $id)
    {

        try {
            $gallery = TaskGallary::find($id);
            if (!$gallery) {
                return response()->json(['error' => 'Gallery not found'], 404);
            }

            // Determine type
            $hasFiles  = $request->hasFile('Image');
            $hasBase64 = $request->filled('Image') && is_array($request->Image);

            if (!$hasFiles && !$hasBase64) {
                return response()->json(['error' => 'No images supplied'], 422);
            }

            // Validation
            if ($hasFiles) {
                $request->validate([
                    'Image'   => 'required|array',
                    'Image.*' => 'image'
                ]);
            } else {
                $request->validate([
                    'Image'   => 'required|array',
                    'Image.*' => 'string'
                ]);
            }

            // Get current images
            $imageUrls = is_array($gallery->Image)
                ? $gallery->Image
                : (json_decode($gallery->Image, true) ?: []);

            $storageDir = public_path('storage/task_gallery');
            if (!file_exists($storageDir)) mkdir($storageDir, 0777, true);

            // a. Multipart Upload
            if ($hasFiles) {
                foreach ($request->file('Image') as $img) {
                    $fname = time() . '_' . uniqid() . '.' . $img->getClientOriginalExtension();
                    $img->move($storageDir, $fname);
                    $imageUrls[] = url('storage/task_gallery/' . $fname);
                }
            }

            // b. Base64 Upload
            if ($hasBase64) {
                foreach ($request->Image as $b64) {
                    if (!preg_match('/^data:image\/(\w+);base64,/', $b64, $m)) {
                        return response()->json(['error' => 'Invalid base64 image'], 422);
                    }
                    $ext = $m[1];
                    $bin = base64_decode(preg_replace('/^data:image\/\w+;base64,/', '', $b64));
                    if ($bin === false) {
                        return response()->json(['error' => 'Invalid base64 data'], 422);
                    }
                    $fname = time() . '_' . uniqid() . '.' . $ext;
                    file_put_contents($storageDir . '/' . $fname, $bin);
                    $imageUrls[] = url('storage/task_gallery/' . $fname);
                }
            }

            $gallery->Image = array_values($imageUrls);
            $gallery->save();

            return response()->json([
                'message' => 'Image(s) added successfully',
                'images'  => $gallery->Image
            ], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (\Throwable $e) {
            Log::error('Add Images Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to add image(s)'], 500);
        }
    }
}
