<?php

namespace App\Http\Controllers;

use App\Models\SuggestionMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class SuggestionMasterController extends Controller
{
    // ─── List All ───────────────────────
    public function index()
    {
        $data = SuggestionMaster::orderBy('sequence', 'asc')->get();
        return response()->json(['status' => true, 'data' => $data]);
    }

    // ─── Create ─────────────────────────
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'suggestion' => 'nullable|string',
            'sequence' => 'nullable|integer',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
        }

        $sequence = $request->input('sequence');

        // Auto-increment if sequence not provided
        if ($sequence === null) {
            $maxSequence = SuggestionMaster::max('sequence');
            $sequence = $maxSequence ? $maxSequence + 1 : 1;
        }

        $data = SuggestionMaster::create([
            'suggestion' => $request->input('suggestion'),
            'sequence'   => $sequence,
        ]);

        return response()->json(['status' => true, 'message' => 'Suggestion created', 'data' => $data]);
    }

    // ─── Get Single ─────────────────────
    public function show($id)
    {
        $data = SuggestionMaster::find($id);
        if (!$data) {
            return response()->json(['status' => false, 'message' => 'Not found'], 404);
        }

        return response()->json(['status' => true, 'data' => $data]);
    }

    // ─── Update ─────────────────────────
    public function update(Request $request, $id)
    {
        $data = SuggestionMaster::find($id);
        if (!$data) {
            return response()->json(['status' => false, 'message' => 'Not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'suggestion' => 'sometimes|nullable|string',
            'sequence'   => 'sometimes|nullable|integer',
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
        }

        $newSequence = $request->input('sequence');
        $oldSequence = $data->sequence;

        // Shift others if sequence changed
        if ($newSequence !== null && $newSequence != $oldSequence) {
            if ($newSequence < $oldSequence) {
                // Moving up: shift down
                SuggestionMaster::where('sequence', '>=', $newSequence)
                    ->where('sequence', '<', $oldSequence)
                    ->increment('sequence');
            } else {
                // Moving down: shift up
                SuggestionMaster::where('sequence', '<=', $newSequence)
                    ->where('sequence', '>', $oldSequence)
                    ->decrement('sequence');
            }

            $data->sequence = $newSequence;
        }

        if ($request->has('suggestion')) {
            $data->suggestion = $request->input('suggestion');
        }

        $data->save();

        return response()->json(['status' => true, 'message' => 'Suggestion updated', 'data' => $data]);
    }

    // ─── Delete ─────────────────────────
    public function destroy($id)
    {
        $data = SuggestionMaster::find($id);
        if (!$data) {
            return response()->json(['status' => false, 'message' => 'Not found'], 404);
        }

        $data->delete();
        return response()->json(['status' => true, 'message' => 'Suggestion deleted']);
    }
}
