<?php

namespace App\Http\Controllers;

use App\Models\StatusMaster;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class StatusMasterController extends Controller
{

    public function index()
    {
        $data = StatusMaster::orderBy('sequence', 'asc')->get();
        return response()->json(['status' => true, 'data' => $data]);
    }

    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'status' => 'nullable|string',
            'sequence' => 'nullable|integer'
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
        }

        $sequence = $request->input('sequence');

        // If sequence not provided, auto-generate as (max + 1)
        if ($sequence === null) {
            $maxSequence = StatusMaster::max('sequence');
            $sequence = $maxSequence ? $maxSequence + 1 : 1;
        }

        $data = StatusMaster::create([
            'status' => $request->input('status'),
            'sequence' => $sequence,
        ]);

        return response()->json(['status' => true, 'message' => 'Status created', 'data' => $data]);
    }

    // ─── Get Single ─────────────────────
    public function show($id)
    {
        $data = StatusMaster::find($id);
        if (!$data) {
            return response()->json(['status' => false, 'message' => 'Not found'], 404);
        }

        return response()->json(['status' => true, 'data' => $data]);
    }



    public function update(Request $request, $id)
    {
        $data = StatusMaster::find($id);
        if (!$data) {
            return response()->json(['status' => false, 'message' => 'Not found'], 404);
        }

        $validator = Validator::make($request->all(), [
            'status' => 'sometimes|nullable|string',
            'sequence' => 'sometimes|nullable|integer'
        ]);

        if ($validator->fails()) {
            return response()->json(['status' => false, 'errors' => $validator->errors()], 422);
        }

        $newSequence = $request->input('sequence');
        $oldSequence = $data->sequence;

        if ($newSequence !== null && $newSequence != $oldSequence) {
            if ($newSequence < $oldSequence) {
                // Shift down: e.g., move from 6 → 2, others from 2..5 should move +1
                StatusMaster::where('sequence', '>=', $newSequence)
                    ->where('sequence', '<', $oldSequence)
                    ->increment('sequence');
            } else {
                // Shift up: e.g., move from 2 → 6, others from 3..6 should move -1
                StatusMaster::where('sequence', '<=', $newSequence)
                    ->where('sequence', '>', $oldSequence)
                    ->decrement('sequence');
            }

            $data->sequence = $newSequence;
        }

        if ($request->has('status')) {
            $data->status = $request->status;
        }

        $data->save();

        return response()->json(['status' => true, 'message' => 'Status updated', 'data' => $data]);
    }

    // ─── Delete ─────────────────────────
    public function destroy($id)
    {
        $data = StatusMaster::find($id);
        if (!$data) {
            return response()->json(['status' => false, 'message' => 'Not found'], 404);
        }

        $data->delete();

        return response()->json(['status' => true, 'message' => 'Status deleted']);
    }
}
