<?php

// app/Http/Controllers/PlannerController.php

namespace App\Http\Controllers;

use App\Models\Planner;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Throwable;

class PlannerController extends Controller
{
    // GET all
    public function index()
    {
        try {
            $data = Planner::with('functionType')->get();
            return response()->json($data);
        } catch (Throwable $e) {
            Log::error('Planner Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to fetch planners'], 500);
        }
    }

    // GET one
    public function show($id)
    {
        try {
            $planner = Planner::with('functionType')->find($id);
            if (!$planner) return response()->json(['message' => 'Not found'], 404);
            return response()->json($planner);
        } catch (Throwable $e) {
            Log::error('Planner Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to fetch planner'], 500);
        }
    }

    // CREATE
    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'desc'    => 'nullable|string',
                'func_id' => 'nullable|exists:function_types,id',
            ]);

            $planner = Planner::create($data);
            return response()->json(['message' => 'Created', 'data' => $planner], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Planner Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to create planner'], 500);
        }
    }

    // UPDATE
    public function update(Request $request, $id)
    {
        try {
            $planner = Planner::find($id);
            if (!$planner) return response()->json(['message' => 'Not found'], 404);

            $data = $request->validate([
                'desc'    => 'sometimes|nullable|string',
                'func_id' => 'sometimes|nullable|exists:function_types,id',
            ]);

            $planner->update($data);
            return response()->json(['message' => 'Updated', 'data' => $planner]);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Planner Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to update planner'], 500);
        }
    }

    // DELETE
    public function destroy($id)
    {
        try {
            $planner = Planner::find($id);
            if (!$planner) return response()->json(['message' => 'Not found'], 404);

            $planner->delete();
            return response()->json(['message' => 'Deleted']);
        } catch (Throwable $e) {
            Log::error('Planner Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to delete planner'], 500);
        }
    }
}
