<?php


namespace App\Http\Controllers;

use App\Models\Noticeboard;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Str;
use Throwable;

class NoticeboardController extends Controller
{

    public function index()
    {
        try {
            $notices = Noticeboard::with('event', 'event.user')->get();
            return response()->json($notices, 200);
        } catch (Throwable $e) {
            Log::error('Noticeboard Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch notices'], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'Ntc_Title'  => 'required|string',
                'NTC_Desc'   => 'nullable|string',
                'Severity'   => 'nullable|integer',
                'event_id'   => 'nullable|exists:events,id',
                'Ntc_S_Dt'   => 'nullable|date',
                'NTC_E_Dt'   => 'nullable|date',
                'NTC_Status' => 'nullable|in:0,1,2,3',
                'NTC_Image'  => 'nullable|array',
                'NTC_Image.*' => 'image',

            ]);

            $imageUrls = [];

            if ($request->hasFile('NTC_Image')) {
                foreach ($request->file('NTC_Image') as $file) {
                    $filename = time() . '_' . uniqid() . '.' . $file->getClientOriginalExtension();
                    $file->move(public_path('storage/noticeboard/images'), $filename);
                    $imageUrls[] = url('storage/noticeboard/images/' . $filename);
                }
            }

            $data['NTC_Image'] = $imageUrls;


            $notice = Noticeboard::create($data);

            return response()->json(['message' => 'Notice created', 'data' => $notice], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Noticeboard Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create notice'], 500);
        }
    }

    public function show($id)
    {
        try {
            $notice = Noticeboard::with('event', 'event.user')->find($id);
            if (!$notice) return response()->json(['message' => 'Notice not found'], 404);
            return response()->json($notice, 200);
        } catch (Throwable $e) {
            Log::error('Noticeboard Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch notice'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $notice = Noticeboard::find($id);
            if (!$notice) return response()->json(['message' => 'Notice not found'], 404);

            $data = $request->validate([
                'Ntc_Title'  => 'sometimes|nullable|string|max:255',
                'NTC_Desc'   => 'sometimes|nullable|string',
                'Severity'   => 'sometimes|nullable|integer',
                'event_id'   => 'sometimes|nullable|exists:events,id',
                'Ntc_S_Dt'   => 'sometimes|nullable|date',
                'NTC_E_Dt'   => 'sometimes|nullable|date',
                'NTC_Status' => 'sometimes|nullable|in:0,1,2,3',
                'NTC_Image'  => 'sometimes|nullable|array',
                'NTC_Image.*' => 'nullable|string',
            ]);

            $imageUrls = is_array($notice->NTC_Image)
                ? $notice->NTC_Image
                : (json_decode($notice->NTC_Image, true) ?: []);

            if ($request->filled('NTC_Image')) {
                $imageUrls = [];
                foreach ($request->input('NTC_Image') as $base64Image) {
                    if (preg_match('/^data:image\/(\w+);base64,/', $base64Image)) {
                        $extension = preg_match('/^data:image\/(\w+);base64,/', $base64Image, $matches) ? $matches[1] : 'png';
                        $base64Image = preg_replace('/^data:image\/\w+;base64,/', '', $base64Image);
                        $base64Image = str_replace(' ', '+', $base64Image);
                        $binary = base64_decode($base64Image);

                        if ($binary === false) {
                            return response()->json(['error' => 'Invalid base64 image'], 422);
                        }

                        $filename = time() . '_' . uniqid() . '.' . $extension;
                        $filepath = public_path('storage/noticeboard/images/' . $filename);
                        file_put_contents($filepath, $binary);
                        $imageUrls[] = url('storage/noticeboard/images/' . $filename);
                    }
                }
            }

            $data['NTC_Image'] = $imageUrls;
            $notice->update($data);
            return response()->json(['message' => 'Notice updated', 'data' => $notice], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Noticeboard Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update notice'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $notice = Noticeboard::find($id);
            if (!$notice) return response()->json(['message' => 'Notice not found'], 404);

            $notice->delete();
            return response()->json(['message' => 'Notice deleted'], 200);
        } catch (Throwable $e) {
            Log::error('Noticeboard Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete notice'], 500);
        }
    }
}
