<?php

namespace App\Http\Controllers;

use App\Models\NewFunctionType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class NewFunctionTypeController extends Controller
{
    public function index()
    {
        try {
            $data = NewFunctionType::with(['eventType', 'parentFunctionType'])->get();
            return response()->json($data);
        } catch (Throwable $e) {
            Log::error('Fetch Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to fetch data'], 500);
        }
    }

    public function show($id)
    {
        try {
            $data = NewFunctionType::with(['eventType', 'parentFunctionType'])->find($id);
            if (!$data) return response()->json(['message' => 'Not found'], 404);
            return response()->json($data);
        } catch (Throwable $e) {
            Log::error('Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to fetch entry'], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $grsh = $request->has('Grsh') ? $request->input('Grsh') : '1';
            $grsrn = $request->has('Grsrn')
                ? $request->input('Grsrn')
                : NewFunctionType::where('Grsh', $grsh)->count() + 1;

            $prefix = str_pad($grsrn, 2, '0', STR_PAD_LEFT);
            $codeFormat = $prefix . '000000';

            $data = [
                'desc'       => $request->input('desc'),
                'ev_type_id' => $request->input('ev_type_id'),
                'func_id'    => $request->input('func_id'),
                'Grsh'       => $grsh,
                'Grsrn'      => (string) $grsrn,
                'Upgrp'      => $request->input('Upgrp', $codeFormat),
                'Grcod'      => $request->input('Grcod', $codeFormat),
            ];

            $type = NewFunctionType::create($data);

            return response()->json(['message' => 'Created', 'data' => $type], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Create Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to create'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $type = NewFunctionType::find($id);
            if (!$type) return response()->json(['message' => 'Not found'], 404);

            $data = $request->validate([
                'desc'       => 'sometimes|nullable|string',
                'ev_type_id' => 'sometimes|nullable|exists:event_types,id',
                'Upgrp'      => 'sometimes|nullable|string',
                'Grcod'      => 'sometimes|nullable|string',
                'Grsrn'      => 'sometimes|nullable|string',
                'Grsh'       => 'sometimes|nullable|string',
                'func_id'    => 'sometimes|nullable|exists:new_function_types,id',
            ]);

            $type->update($data);
            return response()->json(['message' => 'Updated', 'data' => $type]);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to update'], 500);
        }
    }


    public function destroy($id)
    {
        try {
            $type = NewFunctionType::find($id);
            if (!$type) return response()->json(['message' => 'Not found'], 404);

            $type->delete();
            return response()->json(['message' => 'Deleted']);
        } catch (Throwable $e) {
            Log::error('Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to delete'], 500);
        }
    }
}
