<?php

// app/Http/Controllers/HeadFamilyController.php

namespace App\Http\Controllers;

use App\Models\HeadFamily;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Throwable;

class HeadFamilyController extends Controller
{
    public function index()
    {
        try {
            $families = HeadFamily::with(['user', 'relation'])->get();
            return response()->json($families);
        } catch (Throwable $e) {
            Log::error('Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to fetch data'], 500);
        }
    }

    public function show($id)
    {
        try {
            $family = HeadFamily::with(['user', 'relation'])->find($id);
            if (!$family) return response()->json(['message' => 'Not found'], 404);
            return response()->json($family);
        } catch (Throwable $e) {
            Log::error('Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to fetch record'], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'user_id'     => 'nullable|exists:user_masters,id',
                'relation_id' => 'nullable|exists:relations,id',
            ]);

            $family = HeadFamily::create($data);
            return response()->json(['message' => 'Created', 'data' => $family], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to create'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $family = HeadFamily::find($id);
            if (!$family) return response()->json(['message' => 'Not found'], 404);

            $data = $request->validate([
                'user_id'     => 'sometimes|nullable|exists:user_masters,id',
                'relation_id' => 'sometimes|nullable|exists:relations,id',
            ]);

            $family->update($data);
            return response()->json(['message' => 'Updated', 'data' => $family]);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to update'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $family = HeadFamily::find($id);
            if (!$family) return response()->json(['message' => 'Not found'], 404);

            $family->delete();
            return response()->json(['message' => 'Deleted']);
        } catch (Throwable $e) {
            Log::error('Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to delete'], 500);
        }
    }
}
