<?php

// app/Http/Controllers/FunctionTypeController.php

namespace App\Http\Controllers;

use App\Models\FunctionType;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Throwable;

class FunctionTypeController extends Controller
{
    // GET all
    public function index(Request $request)
    {
        try {
            $queryValue = $request->input('query');

            $functionTypes = FunctionType::with('eventType', 'parentFunctionType');

            if ($queryValue) {
                $functionTypes->where(function ($q) use ($queryValue) {
                    $q->where('Upgrp', 'like', $queryValue . '%');
                });
            }

            $data = $functionTypes->get();

            return response()->json(
                $data,
            );
        } catch (\Throwable $e) {
            Log::error('Fetch Error: ' . $e->getMessage());
            return response()->json([
                'status' => 'error',
                'message' => 'Unable to fetch function types',
                'error' => $e->getMessage(),
            ], 500);
        }
    }




    // GET one
    public function show($id)
    {
        try {
            $data = FunctionType::with('eventType')->find($id);
            if (!$data) return response()->json(['message' => 'Not found'], 404);
            return response()->json($data);
        } catch (Throwable $e) {
            Log::error('Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to fetch function type'], 500);
        }
    }




    // public function store(Request $request)
    // {
    //     try {
    //         // Get the next Grsrn value
    //         $nextGrsrn = FunctionType::where('Grsh', 1)->count() + 1;

    //         // Pad the prefix (01, 02, ...) with leading zero
    //         $prefix = str_pad($nextGrsrn, 2, '0', STR_PAD_LEFT);

    //         // Generate the Upgrp and Grcod format
    //         $codeFormat = $prefix . '000000';

    //         $data = [
    //             'desc'       => $request->input('desc'),
    //             'ev_type_id' => $request->input('ev_type_id'),
    //             'Upgrp'      => $codeFormat,
    //             'Grcod'      => $codeFormat,
    //             'Grsrn'      => (string) $nextGrsrn,
    //             'Grsh'       => '1', // Always set to 1
    //             'func_id'    => $request->input('func_id'), // ✅ Optional field

    //         ];

    //         $type = FunctionType::create($data);

    //         return response()->json(['message' => 'Created', 'data' => $type], 201);
    //     } catch (ValidationException $e) {
    //         return response()->json(['errors' => $e->errors()], 422);
    //     } catch (Throwable $e) {
    //         Log::error('Create Error: ' . $e->getMessage());
    //         return response()->json(['error' => 'Unable to create'], 500);
    //     }
    // }



    public function store(Request $request)
    {
        try {
            // Default Grsh
            $grsh = $request->has('Grsh') ? $request->input('Grsh') : '1';
            $grsrn = $request->has('Grsrn')
                ? $request->input('Grsrn')
                : FunctionType::where('Grsh', $grsh)->count() + 1;

            // Format code
            $prefix = str_pad($grsrn, 2, '0', STR_PAD_LEFT);
            $codeFormat = $prefix . '000000';

            $data = [
                'desc'       => $request->input('desc'),
                'ev_type_id' => $request->input('ev_type_id'),
                'func_id'    => $request->input('func_id'),
                'Grsh'       => $grsh,
                'Grsrn'      => (string) $grsrn,
                'Upgrp'      => $request->input('Upgrp', $codeFormat),
                'Grcod'      => $request->input('Grcod', $codeFormat),
            ];

            $type = FunctionType::create($data);

            return response()->json(['message' => 'Created', 'data' => $type], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Create Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to create'], 500);
        }
    }

    // UPDATE
    public function update(Request $request, $id)
    {
        try {
            $type = FunctionType::find($id);
            if (!$type) return response()->json(['message' => 'Not found'], 404);

            $data = $request->validate([
                'desc'       => 'sometimes|nullable|string',
                'ev_type_id' => 'sometimes|nullable|exists:event_types,id',
                'Upgrp'      => 'sometimes|nullable|string',
                'Grcod'      => 'sometimes|nullable|string',
                'Grsrn'      => 'sometimes|nullable|string',
                'Grsh'       => 'sometimes|nullable|string',
                'func_id'    => 'sometimes|nullable|exists:function_types,id',

            ]);

            $type->update($data);
            return response()->json(['message' => 'Updated', 'data' => $type]);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to update'], 500);
        }
    }

    // DELETE
    public function destroy($id)
    {
        try {
            $type = FunctionType::find($id);
            if (!$type) return response()->json(['message' => 'Not found'], 404);

            $type->delete();
            return response()->json(['message' => 'Deleted']);
        } catch (Throwable $e) {
            Log::error('Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Unable to delete'], 500);
        }
    }
}
