<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\EventType;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class EventTypeController extends Controller
{
    public function index()
    {
        try {
            return response()->json(EventType::all(), 200);
        } catch (Throwable $e) {
            Log::error('EventType Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch'], 500);
        }
    }

    public function show($id)
    {
        try {
            $event = EventType::find($id);
            if (!$event) return response()->json(['message' => 'Not found'], 404);
            return response()->json($event, 200);
        } catch (Throwable $e) {
            Log::error('EventType Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch'], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'event_name' => 'nullable|string',
            ]);

            $event = EventType::create($data);
            return response()->json(['message' => 'Created', 'data' => $event], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('EventType Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $event = EventType::find($id);
            if (!$event) return response()->json(['message' => 'Not found'], 404);

            $data = $request->validate([
                'event_name' => 'sometimes|nullable|string',
            ]);

            $event->update($data);
            return response()->json(['message' => 'Updated', 'data' => $event], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('EventType Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Update failed'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $event = EventType::find($id);
            if (!$event) return response()->json(['message' => 'Not found'], 404);

            $event->delete();
            return response()->json(['message' => 'Deleted'], 200);
        } catch (Throwable $e) {
            Log::error('EventType Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Delete failed'], 500);
        }
    }
}
