<?php

namespace App\Http\Controllers;

use App\Models\EventMember;
use App\Models\UserMaster;
use Illuminate\Http\Request;
use Illuminate\Validation\ValidationException;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;
use Throwable;

class EventMemberController extends Controller
{
    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'event_id'   => 'nullable|exists:events,id',
                'user_id'    => 'nullable|array',
                'user_id.*'  => 'numeric|exists:user_masters,id',
                'EM_Invite'  => 'nullable|date',
                'EM_Accept'  => 'nullable|date',
                'EM_Status'  => 'nullable|in:0,1,2,3,4',
            ]);

            $userArray = collect($data['user_id'])->map(function ($id) {
                return ['user_id' => $id, 'UM_Status' => 0];
            });

            $data['user_id'] = $userArray;
            $data['EM_Invite'] = $data['EM_Invite'] ?? Carbon::now();

            if (($data['EM_Status'] ?? null) == 1 && empty($data['EM_Accept'])) {
                $data['EM_Accept'] = Carbon::now();
            }

            $member = EventMember::create($data);
            $member->setAttribute('users', $member->getUsers());

            return response()->json([
                'message' => 'Event member created',
                'data' => $member
            ], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create event member'], 500);
        }
    }

    public function index()
    {
        try {
            $members = EventMember::with('event')->get();

            $members->each(function ($member) {
                $original = $member->user_id;

                // ✅ Normalize user_detail to always be an array of { user_id, UM_Status }
                $userDetail = collect($original)->map(function ($u) {
                    if (is_array($u)) {
                        return [
                            'user_id'   => $u['user_id'],
                            'UM_Status' => $u['UM_Status'] ?? 0,
                        ];
                    } else {
                        return [
                            'user_id'   => $u,
                            'UM_Status' => 0,
                        ];
                    }
                });

                $member->setAttribute('user_detail', $userDetail);

                $member->setAttribute('user_id', $userDetail->pluck('user_id')->values());

                $users = $userDetail->map(function ($userData) {
                    $user = \App\Models\UserMaster::find($userData['user_id']);
                    if ($user) {
                        $user->UM_Status = $userData['UM_Status'];
                        return $user;
                    }
                    return null;
                })->filter()->values();

                $member->setAttribute('users', $users);
            });

            return response()->json($members, 200);
        } catch (\Throwable $e) {
            Log::error('Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch event members'], 500);
        }
    }


    public function show($id)
    {
        try {
            $member = EventMember::find($id);
            if (!$member) {
                return response()->json(['message' => 'Event member not found'], 404);
            }

            $member->setAttribute('user_id', collect($member->user_id)->pluck('user_id')->values());
            $member->setAttribute('users', $member->getUsers());


            return response()->json($member, 200);
        } catch (Throwable $e) {
            Log::error('Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch event member'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $member = EventMember::find($id);
            if (!$member) {
                return response()->json(['message' => 'Event member not found'], 404);
            }

            $data = $request->validate([
                'event_id'     => 'sometimes|nullable|exists:events,id',
                'EM_Invite'    => 'sometimes|nullable|date',
                'EM_Accept'    => 'sometimes|nullable|date',
                'EM_Status'    => 'sometimes|nullable|in:0,1,2,3,4',

                'user_id'      => 'sometimes|array',
                'user_id.*'    => 'numeric|exists:user_masters,id',

                'user_status'  => 'sometimes|array',
                'user_status.*.user_id'   => 'sometimes|nullable|numeric|exists:user_masters,id',
                'user_status.*.UM_Status' => 'sometimes|nullable|in:0,1,2,3,4',
            ]);

            // Replace entire user list if provided
            if (isset($data['user_id'])) {
                $newUsers = collect($data['user_id'])->map(function ($uid) {
                    return [
                        'user_id' => $uid,
                        'UM_Status' => 0
                    ];
                });
                $member->user_id = $newUsers->toArray();
            }

            // Only update UM_Status of existing users if user_status is provided
            if (isset($data['user_status'])) {
                $existingUsers = collect($member->user_id);
                $updates = collect($data['user_status'])->keyBy('user_id');

                $updatedUsers = $existingUsers->map(function ($user) use ($updates) {
                    if (isset($updates[$user['user_id']])) {
                        $user['UM_Status'] = $updates[$user['user_id']]['UM_Status'];
                    }
                    return $user;
                });

                $member->user_id = $updatedUsers->toArray();
            }

            // Fill timestamps if needed
            if (empty($data['EM_Invite'])) {
                $data['EM_Invite'] = Carbon::now();
            }

            if (($data['EM_Status'] ?? null) == 1 && empty($data['EM_Accept'])) {
                $data['EM_Accept'] = Carbon::now();
            }

            // Apply remaining fields
            $member->event_id   = $data['event_id']   ?? $member->event_id;
            $member->EM_Invite  = $data['EM_Invite']  ?? $member->EM_Invite;
            $member->EM_Accept  = $data['EM_Accept']  ?? $member->EM_Accept;
            $member->EM_Status  = $data['EM_Status']  ?? $member->EM_Status;

            $member->save();

            $member->setAttribute('users', $member->getUsers());

            return response()->json([
                'message' => 'Event member updated',
                'data' => $member
            ], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update event member'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $member = EventMember::find($id);
            if (!$member) {
                return response()->json(['message' => 'Event member not found'], 404);
            }

            $member->delete();
            return response()->json(['message' => 'Event member deleted'], 200);
        } catch (Throwable $e) {
            Log::error('Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete event member'], 500);
        }
    }
}
