<?php

namespace App\Http\Controllers;

use App\Models\Event;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class EventController extends Controller
{

    public function index()
    {
        try {
            $events = Event::with(['user', 'eventType'])->get();
            return response()->json($events, 200);
        } catch (Throwable $e) {
            Log::error('Event index error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch events'], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            $data = $request->validate([
                'Ev_Name'         => 'required|string',
                'Ev_Date'         => 'nullable|date',
                'user_id'         => 'nullable|exists:user_masters,id',
                'ev_type_id'      => 'nullable|exists:event_types,id',
                'image'           => 'nullable|array',
                'image.*'         => 'image',
                'expected_guests' => 'nullable|integer',
                'location'        => 'nullable|string',
                'description'     => 'nullable|string',
            ]);

            $imageUrls = [];

            if ($request->hasFile('image')) {
                foreach ($request->file('image') as $img) {
                    $name = time() . '_' . uniqid() . '.' . $img->getClientOriginalExtension();
                    $img->move(public_path('storage/event_images'), $name);
                    $imageUrls[] = url('storage/event_images/' . $name);
                }
            }

            $data['image'] = $imageUrls;

            $event = Event::create($data);

            return response()->json(['message' => 'Event created', 'event' => $event], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Event Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create event'], 500);
        }
    }

    public function show($id)
    {
        try {
            $event = Event::with(['user', 'eventType'])->find($id);
            if (!$event) {
                return response()->json(['message' => 'Event not found'], 404);
            }

            return response()->json($event, 200);
        } catch (Throwable $e) {
            Log::error('Event show error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch event'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $event = Event::find($id);
            if (!$event) return response()->json(['message' => 'Event not found'], 404);

            $data = $request->validate([
                'Ev_Name'         => 'sometimes|nullable|string',
                'Ev_Date'         => 'sometimes|nullable|date',
                'user_id'         => 'sometimes|nullable|exists:user_masters,id',
                'ev_type_id'      => 'sometimes|nullable|exists:event_types,id',
                'image'           => 'sometimes|nullable|array',
                'image.*'         => 'string|nullable',
                'expected_guests' => 'sometimes|nullable|integer',
                'location'        => 'sometimes|nullable|string',
                'description'     => 'sometimes|nullable|string',
            ]);

            $imageUrls = is_array($event->image)
                ? $event->image
                : (json_decode($event->image, true) ?: []);

            if ($request->filled('image')) {
                $imageUrls = [];

                foreach ($request->input('image') as $b64) {
                    if (preg_match('/^data:image\/(\w+);base64,/', $b64, $matches)) {
                        $ext = $matches[1];
                        $bin = base64_decode(preg_replace('/^data:image\/\w+;base64,/', '', $b64));
                        if ($bin === false) return response()->json(['error' => 'Invalid base64 image'], 422);

                        $filename = time() . '_' . uniqid() . '.' . $ext;
                        $path = public_path('storage/event_images/' . $filename);
                        file_put_contents($path, $bin);

                        $imageUrls[] = url('storage/event_images/' . $filename);
                    }
                }
            }

            $data['image'] = $imageUrls;

            $event->update($data);

            return response()->json(['message' => 'Event updated', 'event' => $event], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('Event Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update event'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $event = Event::find($id);
            if (!$event) {
                return response()->json(['message' => 'Event not found'], 404);
            }

            $event->delete();
            return response()->json(['message' => 'Event deleted'], 200);
        } catch (Throwable $e) {
            Log::error('Event delete error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete event'], 500);
        }
    }
}
