<?php

namespace App\Http\Controllers;

use App\Models\EvTask;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class EvTaskController extends Controller
{
    public function index()
    {
        try {
            $tasks = EvTask::with('event')->get();

            $tasks->each(function ($task) {
                $task->setAttribute('fun_type_data', $task->fun_type_data);
            });

            return response()->json($tasks, 200);
        } catch (Throwable $e) {
            Log::error('EvTask Index Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch tasks'], 500);
        }
    }

    public function store(Request $request)
    {
        try {
            // ✅ Ensure fun_type is a single integer (not an array)
            if (is_array($request->fun_type)) {
                $request->merge([
                    'fun_type' => $request->fun_type[0] ?? null
                ]);
            }

            $data = $request->validate([
                'event_id'    => 'nullable|exists:events,id',
                'Task_Name'   => 'required|string',
                'Task_Date'   => 'nullable|date',
                'Task_Till'   => 'nullable|date',
                'Task_Venue'  => 'nullable|string',
                'Task_GPS'    => 'nullable|string',
                'description' => 'nullable|string',
                'fun_type'    => 'nullable|exists:function_types,id',
                'Upgrp'       => 'nullable|string',
                'Grcod'       => 'nullable|string',
                'Grsrn'       => 'nullable|string',
                'Grsh'        => 'nullable|string',
                'Status'      => 'nullable|integer|in:0,1,2,3', // ✅ only allow 0 or 1

            ]);

            $task = EvTask::create($data);
            $task->setAttribute('fun_type_data', $task->fun_type_data);

            return response()->json(['message' => 'Task created', 'data' => $task], 201);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('EvTask Store Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to create task'], 500);
        }
    }

    public function show($id)
    {
        try {
            $task = EvTask::with('event')->find($id);
            if (!$task) return response()->json(['message' => 'Task not found'], 404);

            $task->setAttribute('fun_type_data', $task->fun_type_data);

            return response()->json($task, 200);
        } catch (Throwable $e) {
            Log::error('EvTask Show Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to fetch task'], 500);
        }
    }

    public function update(Request $request, $id)
    {
        try {
            $task = EvTask::find($id);
            if (!$task) return response()->json(['message' => 'Task not found'], 404);

            // ✅ Ensure fun_type is a single integer
            if ($request->has('fun_type') && is_array($request->fun_type)) {
                $request->merge([
                    'fun_type' => $request->fun_type[0] ?? null
                ]);
            }

            $data = $request->validate([
                'event_id'    => 'sometimes|nullable|exists:events,id',
                'Task_Name'   => 'sometimes|nullable|string',
                'Task_Date'   => 'sometimes|nullable|date',
                'Task_Till'   => 'sometimes|nullable|date',
                'Task_Venue'  => 'sometimes|nullable|string',
                'Task_GPS'    => 'sometimes|nullable|string',
                'description' => 'sometimes|nullable|string',
                'fun_type'    => 'sometimes|nullable|exists:function_types,id',
                'Upgrp'       => 'sometimes|nullable|string',
                'Grcod'       => 'sometimes|nullable|string',
                'Grsrn'       => 'sometimes|nullable|string',
                'Grsh'        => 'sometimes|nullable|string',
                'Status'      => 'sometimes|nullable|integer|in:0,1,2,3', // ✅ Added

            ]);

            $task->update($data);
            $task->setAttribute('fun_type_data', $task->fun_type_data);

            return response()->json(['message' => 'Task updated', 'data' => $task], 200);
        } catch (ValidationException $e) {
            return response()->json(['errors' => $e->errors()], 422);
        } catch (Throwable $e) {
            Log::error('EvTask Update Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to update task'], 500);
        }
    }

    public function destroy($id)
    {
        try {
            $task = EvTask::find($id);
            if (!$task) return response()->json(['message' => 'Task not found'], 404);

            $task->delete();
            return response()->json(['message' => 'Task deleted'], 200);
        } catch (Throwable $e) {
            Log::error('EvTask Delete Error: ' . $e->getMessage());
            return response()->json(['error' => 'Failed to delete task'], 500);
        }
    }
}
