<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use App\Models\UserMaster;
use App\Mail\ForgotPasswordMail;
use Carbon\Carbon;
use Exception;

class AuthController extends Controller
{
    // Step 1: Send OTP
    public function sendOtp(Request $request)
    {
        try {
            $request->validate([
                'email' => 'required|email|exists:user_masters,email'
            ]);

            $otp = rand(100000, 999999);

            DB::table('password_resets')->updateOrInsert(
                ['email' => $request->email],
                ['otp' => $otp, 'created_at' => Carbon::now()]
            );

            Mail::to($request->email)->send(new ForgotPasswordMail($otp));

            return response()->json(['message' => 'OTP sent successfully!'], 200);
        } catch (Exception $e) {
            return response()->json(['error' => 'Failed to send OTP: ' . $e->getMessage()], 500);
        }
    }

    // Step 2: Verify OTP
    public function verifyOtp(Request $request)
    {
        try {
            $request->validate(['otp' => 'required|digits:6']);

            $otpRecord = DB::table('password_resets')
                ->where('otp', $request->otp)
                ->where('created_at', '>=', Carbon::now()->subMinutes(10))
                ->first();

            if (!$otpRecord) {
                return response()->json(['error' => 'Invalid or expired OTP!'], 400);
            }

            return response()->json(['message' => 'OTP verified successfully!'], 200);
        } catch (Exception $e) {
            return response()->json(['error' => 'OTP verification failed: ' . $e->getMessage()], 500);
        }
    }

    // Step 3: Reset Password
    public function resetPassword(Request $request)
    {
        try {
            $request->validate([
                'password' => 'required|string',
                'confirm_password' => 'required|string|same:password'
            ]);

            $otpRecord = DB::table('password_resets')
                ->where('created_at', '>=', Carbon::now()->subMinutes(10))
                ->latest('created_at')
                ->first();

            if (!$otpRecord) {
                return response()->json(['error' => 'OTP expired or not found.'], 400);
            }

            $user = UserMaster::where('email', $otpRecord->email)->first();
            if (!$user) {
                return response()->json(['error' => 'User not found'], 404);
            }

            $user->password = $request->password; // model mutator will hash it
            $user->save();

            DB::table('password_resets')->where('email', $otpRecord->email)->delete();

            return response()->json(['message' => 'Password reset successfully!'], 200);
        } catch (Exception $e) {
            return response()->json(['error' => 'Password reset failed: ' . $e->getMessage()], 500);
        }
    }
}
